import React, { useEffect, useState, useCallback, useMemo } from "react";
import dynamic from "next/dynamic";
import { RiArrowUpSLine } from "react-icons/ri";
import Head from "next/head";
import parse from "html-react-parser";
import Script from "next/script";
import { useLayout } from "../../contexts/LayoutContext"; // ← ADD THIS IMPORT

// Dynamic imports with loading states - reduces initial bundle
const MainMenu = dynamic(() => import("./main-menu"), {
  loading: () => <div style={{ height: "45px" }} />,
  ssr: true,
});

const TopNavbar = dynamic(() => import("./top-navbar"), {
  loading: () => <div style={{ height: "35px" }} />,
  ssr: true,
});

const BottomMenu = dynamic(() => import("./bottom-menu"), {
  ssr: true,
});

const MainFooter = dynamic(() => import("./main-footer"), {
  ssr: true,
});

const GoogleTranslate = dynamic(() => import("./GoogleTranslate"), {
  ssr: false,
  loading: () => <div style={{ height: "40px" }} />,
});

// ← REMOVE all the props, only keep 'children'
export default function MainLayout({ children }) {
  const [scrollBtn, setScrollBtn] = useState(false);

  // ← GET DATA FROM CONTEXT INSTEAD OF PROPS
  const {
    mainMenu: menuItems,
    mainMenuBtn,
    mobileMainMenu,
    topMenu,
    mobileTopMenu,
    bottomMenu,
    footer,
    globalTags,
    mobileLayout,
  } = useLayout();

  // Optimized scroll handler with throttling
  useEffect(() => {
    let timeoutId;

    const handleScroll = () => {
      if (timeoutId) {
        clearTimeout(timeoutId);
      }

      timeoutId = setTimeout(() => {
        setScrollBtn(window.scrollY > 400);
      }, 100);
    };

    window.addEventListener("scroll", handleScroll, { passive: true });

    return () => {
      window.removeEventListener("scroll", handleScroll);
      if (timeoutId) {
        clearTimeout(timeoutId);
      }
    };
  }, []);

  const scrollToTop = useCallback(() => {
    window.scrollTo({ top: 0, left: 0, behavior: "smooth" });
  }, []);

  const phoneNumber = useMemo(() => mobileLayout?.bottom || "", [mobileLayout]);

  return (
    <>
      <Head>
        <meta
          property="og:site_name"
          content="Emergency Room: SignatureCare Emergency Center 24-Hour ER"
        />
        <meta property="og:locale" content="en_US" />
        <meta property="og:type" content="article" />
        <meta
          property="article:publisher"
          content="https://www.facebook.com/SignatureCareER/"
        />
        <meta name="geo.placename" content="Houston" />
        <meta name="geo.position" content="29.744225;-95.3918991" />
        <meta name="geo.region" content="United States (US)" />

        <link rel="preconnect" href="https://translate.googleapis.com" />
        <link rel="preconnect" href="https://translate.google.com" />
      </Head>

      {globalTags?.first && (
        <div suppressHydrationWarning>{parse(globalTags.first)}</div>
      )}

      <GoogleTranslate />

      <div className="page-container">
        <div className="sticky-md-top">
          <div className="d-none d-md-block">
            <TopNavbar menuItems={topMenu} style={{ height: "35px" }} />
            <MainMenu
              menuItems={menuItems}
              mainMenuBtn={mainMenuBtn}
              style={{ height: "45px" }}
            />
          </div>

          <div className="d-block d-md-none">
            <TopNavbar menuItems={mobileTopMenu} style={{ height: "35px" }} />
            <MainMenu
              menuItems={mobileMainMenu}
              mainMenuBtn={mainMenuBtn}
              style={{ height: "45px" }}
            />
          </div>
        </div>

        <div className="content-wrap" style={{ minHeight: "700px" }}>
          {children}
        </div>

        {scrollBtn && (
          <button
            onClick={scrollToTop}
            aria-label="Scroll to top"
            style={{
              position: "fixed",
              bottom: "5px",
              left: "40px",
              padding: "5px 10px",
              backgroundColor: "black",
              color: "white",
              borderRadius: "5px 5px 0 0",
              cursor: "pointer",
              zIndex: "1",
              border: "none",
            }}
          >
            <RiArrowUpSLine className="fa-2x" aria-hidden="true" />
          </button>
        )}

        <footer className="footer text-white">
          <MainFooter footer={footer} />
        </footer>

        <BottomMenu menuItems={bottomMenu} />
      </div>

      <nav
        className="d-lg-none navbar fixed-bottom navbar-dark p-1"
        style={{ backgroundColor: "hsl(0, 100%, 40%)" }}
        aria-label="Mobile call navigation"
      >
        <div className="container-fluid">
          <a
            className="navbar-brand mx-auto"
            href={`tel:${phoneNumber}`}
            aria-label={`Call us at ${phoneNumber}`}
          >
            Call us now!
          </a>
        </div>
      </nav>

      {globalTags?.second && (
        <div suppressHydrationWarning>{parse(globalTags.second)}</div>
      )}
    </>
  );
}
