import { useRouter } from 'next/router';
import React, { useRef, useState } from 'react'
import { Form, Row, Col, Button, Spinner } from 'react-bootstrap';
import { ToastContainer, toast } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';
import http from '../utils/http';
import ReCAPTCHA from 'react-google-recaptcha'

export default function ContactForm(props) {
    const router = useRouter();
    const [inputs, setInputs] = useState({});
    const [validated, setValidated] = useState(false);
    const [isClicked, setIsClicked] = useState(false);
    const location = props.location;
    const [disabled, setDisabled] = useState(true);
    const recaptchaRef = useRef(null);

    const handleInput = (event) => {
        const name = event.target.name;
        const value = event.target.value;
        setInputs(values => ({ ...values, [name]: value }));
    }

    const recaptHandle = () => {
        setDisabled(false);
    }

    const handleSubmit = async (event) => {
        event.preventDefault();
        const form = event.currentTarget;

        if (form.checkValidity() === false) {
            event.stopPropagation();
            setValidated(true);
            return false;
        }

        const colorData = new FormData(event.target);

        if (colorData.get("favorite_color") !== "") {
            // Form submission is spam
            return;
        }

        setIsClicked(true);

        const formData = new FormData();
        formData.append('fname', inputs.fname);
        formData.append('lname', inputs.lname);
        formData.append('phone', inputs.phone);
        formData.append('email', inputs.email);
        formData.append('body', inputs.body);

        if (location !== undefined) {
            formData.append('review', location);
        }

        // for (var pair of formData.entries()) {
        //     console.log(pair[0] + ', ' + pair[1]);
        // }

        http.post('/mail/contact-us', formData).then(res => {
            // console.log(res.data)
            toast.success(res.data, {
                position: "top-right",
                autoClose: 5000,
                hideProgressBar: false,
                closeOnClick: true,
                pauseOnHover: true,
                draggable: true,
                progress: undefined,
            });

            setInputs({});

            recaptchaRef.current.reset();
            setIsClicked(false);
            setDisabled(true);
            setValidated(false);
            if (location !== undefined) {
                router.push('/review-thanks');
            }
        }).catch(error => {
            setIsClicked(false);
        })
    };

    return (
        <>
            <ToastContainer />

            <Form noValidate validated={validated} onSubmit={handleSubmit}>
                <Row>
                    <Form.Group as={Col} md="6" controlId="fname" className="mb-3">
                        <Form.Label>First name</Form.Label>
                        <Form.Control
                            name="fname" value={inputs.fname || ''} onChange={handleInput}
                            required
                            type="text"
                            placeholder="First name"
                        />
                        <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                        <Form.Control.Feedback type="invalid">This field is required</Form.Control.Feedback>
                    </Form.Group>
                    <Form.Group as={Col} md="6" controlId="lname" className="mb-3">
                        <Form.Label>Last name</Form.Label>
                        <Form.Control
                            name="lname" value={inputs.lname || ''} onChange={handleInput}
                            required
                            type="text"
                            placeholder="Last name"
                        />
                        <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                        <Form.Control.Feedback type="invalid">This field is required</Form.Control.Feedback>
                    </Form.Group>
                </Row>
                <Row>
                    <Form.Group as={Col} md="6" controlId="phone" className="mb-3">
                        <Form.Label>Phone Number</Form.Label>
                        <Form.Control
                            name="phone" value={inputs.phone || ''} onChange={handleInput}
                            required
                            type="text"
                            placeholder="Phone Number"
                        />
                        <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                        <Form.Control.Feedback type="invalid">This field is required</Form.Control.Feedback>
                    </Form.Group>
                    <Form.Group as={Col} md="6" controlId="email" className="mb-3">
                        <Form.Label>Email</Form.Label>
                        <Form.Control
                            name="email" value={inputs.email || ''} onChange={handleInput}
                            required
                            type="email"
                            placeholder="Email"
                        />
                        <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                        <Form.Control.Feedback type="invalid">Invalid email address!</Form.Control.Feedback>
                    </Form.Group>
                </Row>

                <Row>
                    <Form.Group as={Col} md="12" className="mb-3" controlId="exampleForm.ControlTextarea1">
                        <Form.Label>Reason for Contacting Us <span style={{ color: 'hsl(0,100%,50%)' }}>*</span></Form.Label>
                        <Form.Control name="body" value={inputs.body || ''} onChange={handleInput} as="textarea" rows={5} required placeholder="Type here..." />
                    </Form.Group>
                    <Form.Control.Feedback type="invalid">This field is required</Form.Control.Feedback>
                </Row>

                <Form.Control type="hidden" name="favorite_color" value="" />

                <Form.Group className="mb-3" controlId="formConfirm">
                    <ReCAPTCHA size="normal"
                        ref={recaptchaRef}
                        sitekey={process.env.NEXT_PUBLIC_RECAPTCHA_SITE_KEY}
                        onChange={recaptHandle}
                    />
                </Form.Group>

                <div className="d-grid gap-2">
                    {!isClicked &&
                        <Button disabled={disabled} type="submit" variant="danger" size="lg" className='btn btn-block' >Submit</Button>
                    }

                    {isClicked &&
                        <Button type="submit" variant="danger" size="lg" className='btn btn-block' disabled>
                            <Spinner as="span" animation="border" size="sm" role="status" aria-hidden="true" />
                            <span> Sending...</span>
                        </Button>
                    }
                </div>

            </Form>
        </>
    )
}
