import React, { useEffect, useState, useCallback, useMemo } from "react";
import { Card, ListGroup } from "react-bootstrap";
import {
  GoogleMap,
  useLoadScript,
  MarkerF,
  InfoWindowF,
} from "@react-google-maps/api";
import laravelURL from "../utils/laravel-url";
import Link from "next/link";
import Image from "next/image";

const containerStyle = {
  maxWidth: "100%",
  height: "600px",
  paddingTop: "0",
};

// ✅ Move libraries outside component to prevent re-renders
const libraries = ["places"];

export default function MapHome({ locations = [] }) {
  // ✅ Add libraries array to prevent re-loading
  const { isLoaded, loadError } = useLoadScript({
    id: "google-map-script",
    googleMapsApiKey: process.env.NEXT_PUBLIC_API_KEY,
    libraries, // ← Use constant libraries array
  });

  const [filterMap, setFilterMap] = useState([]);
  const [cardData, setCardData] = useState(null);
  const [show, setShow] = useState(false);
  const [map, setMap] = useState(null);
  const [initialBounds, setInitialBounds] = useState(null);
  const [mapHeight, setMapHeight] = useState(containerStyle.height);

  // ✅ Memoize filtered locations to prevent recalculation
  const filteredLocations = useMemo(() => {
    return (
      locations?.filter(
        (location) => location.name !== "Greater Texas Division"
      ) || []
    );
  }, [locations]);

  useEffect(() => {
    if (window.innerWidth < 768) {
      setMapHeight("400px");
    }
  }, []);

  useEffect(() => {
    if (filteredLocations.length > 0) {
      setFilterMap(filteredLocations);
    }
  }, [filteredLocations]);

  useEffect(() => {
    if (map && filterMap.length > 0 && isLoaded) {
      const bounds = new window.google.maps.LatLngBounds();

      filterMap.forEach((location) => {
        if (location.latitude && location.longitude) {
          bounds.extend(
            new window.google.maps.LatLng(location.latitude, location.longitude)
          );
        }
      });

      setInitialBounds(bounds);

      const bottomPadding = window.innerWidth < 768 ? 0 : 100;

      map.fitBounds(bounds, {
        bottom: bottomPadding,
        left: 0,
        right: 0,
        top: 0,
      });
    }
  }, [filterMap, isLoaded, map]);

  // ✅ Use useCallback to prevent function recreation
  const onMapLoad = useCallback((map) => {
    setMap(map);
  }, []);

  const hoverMarker = useCallback((location) => {
    setCardData(location);
    setShow(true);
  }, []);

  const unhoverMarker = useCallback(() => {
    setShow(false);
    if (map && initialBounds) {
      map.fitBounds(initialBounds, { bottom: 100, left: 0, right: 0, top: 0 });
    }
  }, [map, initialBounds]);

  // ✅ Memoize map options to prevent re-renders
  const mapOptions = useMemo(
    () => ({
      streetViewControl: false,
      zoomControlOptions: {
        position: isLoaded ? window.google.maps.ControlPosition.LEFT_TOP : 0,
      },
    }),
    [isLoaded]
  );

  // ✅ Handle load error
  if (loadError) {
    return (
      <div
        className="d-flex justify-content-center align-items-center"
        style={{ height: mapHeight }}
      >
        <p className="text-danger">Error loading maps</p>
      </div>
    );
  }

  // ✅ Better loading state
  if (!isLoaded) {
    return (
      <div
        className="d-flex justify-content-center align-items-center bg-light"
        style={{ height: mapHeight }}
      >
        <div className="spinner-border text-primary" role="status">
          <span className="visually-hidden">Loading map...</span>
        </div>
      </div>
    );
  }

  return (
    <div aria-label="Interactive emergency room locations map">
      <GoogleMap
        mapContainerStyle={{ ...containerStyle, height: mapHeight }}
        onLoad={onMapLoad}
        onClick={unhoverMarker}
        options={mapOptions}
      >
        {filterMap?.map((location, index) => (
          <MarkerF
            onMouseOver={() => hoverMarker(location)}
            onClick={() => hoverMarker(location)}
            key={location.id || index} // ✅ Use location.id if available
            position={{
              lat: parseFloat(location.latitude),
              lng: parseFloat(location.longitude),
            }}
            icon={{
              url: "/assets/signaturecare-mini.webp",
              scaledSize: isLoaded ? new google.maps.Size(25, 25) : undefined,
            }}
          >
            {show && cardData && cardData.name === location.name && (
              <InfoWindowF
                position={{
                  lat: parseFloat(cardData.latitude),
                  lng: parseFloat(cardData.longitude),
                }}
                onCloseClick={unhoverMarker}
                options={{
                  maxWidth: 300,
                  pixelOffset: new google.maps.Size(0, -10), // ← Move close button up
                }}
              >
                <div>
                  {/* Large Screen */}
                  <Card
                    className="overflow-hidden d-none d-lg-block border-0"
                    style={{ width: "280px" }}
                  >
                    {cardData.img && (
                      <Image
                        src={laravelURL + "/storage/" + cardData.img}
                        alt={cardData.name}
                        width={280}
                        height={160}
                        sizes="280px"
                        quality={80}
                        className="object-fit-cover"
                      />
                    )}
                    <Card.Body className="p-3">
                      <Card.Title className="fs-6 fw-bold mb-2">
                        {cardData.name}
                        {cardData.status === 2 && (
                          <span className="badge bg-danger ms-2">
                            coming soon
                          </span>
                        )}
                      </Card.Title>

                      <div className="d-grid gap-2 mb-3">
                        <Link
                          href={cardData.google || "#"}
                          className="btn btn-danger btn-sm fw-bold"
                          target="_blank"
                          rel="noopener noreferrer"
                        >
                          📍 Get Directions
                        </Link>
                      </div>

                      <div className="mb-2">
                        <div className="text-muted small fw-bold mb-1">
                          ADDRESS
                        </div>
                        <div className="small fw-semibold">
                          {cardData.address}
                        </div>
                      </div>

                      <div>
                        <div className="text-muted small fw-bold mb-1">
                          PHONE
                        </div>
                        <Link
                          href={"tel:" + cardData.tel}
                          className="text-danger text-decoration-none fw-bold small"
                        >
                          📞 {cardData.tel}
                        </Link>
                      </div>
                    </Card.Body>
                  </Card>

                  {/* Small Screen */}
                  <Card
                    className="overflow-hidden border-0 d-lg-none"
                    style={{ width: "260px" }}
                  >
                    {cardData.img && (
                      <Image
                        src={laravelURL + "/storage/" + cardData.img}
                        alt={cardData.name}
                        width={260}
                        height={140}
                        sizes="260px"
                        quality={80}
                        className="object-fit-cover"
                      />
                    )}
                    <Card.Body className="p-2">
                      <Card.Title className="fs-6 fw-bold mb-2">
                        {cardData.name}
                        {cardData.status === 2 && (
                          <span className="badge bg-danger ms-1 small">
                            coming soon
                          </span>
                        )}
                      </Card.Title>

                      <div className="d-grid gap-1">
                        <Link
                          href={cardData.google || "#"}
                          className="btn btn-danger btn-sm fw-bold"
                          target="_blank"
                          rel="noopener noreferrer"
                        >
                          📍 Directions
                        </Link>

                        <div className="small fw-semibold text-muted">
                          {cardData.address}
                        </div>

                        <Link
                          href={"tel:" + cardData.tel}
                          className="btn btn-outline-danger btn-sm fw-bold"
                        >
                          📞 {cardData.tel}
                        </Link>
                      </div>
                    </Card.Body>
                  </Card>
                </div>
              </InfoWindowF>
            )}
          </MarkerF>
        ))}
      </GoogleMap>
    </div>
  );
}
