import React, { useState } from "react";
import Swal from "sweetalert2";
import withReactContent from "sweetalert2-react-content";
import moment from "moment";
import Pagination from "react-js-pagination";
import http from "../../components/utils/http";
import laravelURL from "../../components/utils/laravel-url";
import Link from "next/link";
import Head from "next/head";
import Image from "next/image";
// import useSWR from 'swr'
import mainURL from "../../components/utils/main-url";
import { useRouter } from "next/router";
import { getLayoutData } from "../../components/utils/getLayoutData";

export default function BlogPage({ blogList }) {
  const router = useRouter();
  const [blogs, setBlogs] = useState(blogList.data);
  const [activePage, setActivePage] = useState(1);
  const [total, setTotal] = useState(blogList.total);
  const [perPage, setPerPage] = useState(blogList.per_page);

  const MySwal = withReactContent(Swal);

  const fetchBlogs = async (pageNumber) => {
    const res = await http.get(`/blogs/categories?page=${pageNumber}`);
    const data = await res.data;
    setBlogs(data.data);
    setTotal(data.total);
    setPerPage(data.per_page);
    setActivePage(data.current_page);
    router.push("/blog?page=" + pageNumber, undefined, { shallow: true });
  };

  const handlePageChange = (pageNumber) => {
    // console.log(`active page is ${pageNumber}`);
    fetchBlogs(pageNumber);
  };

  const getDescription = (description) => {
    return { __html: description.replace(/<[^>]+>/g, "").slice(0, 169) };
  };

  return (
    <>
      <Head>
        <title>Blog | SignatureCare Emergency Center</title>
        <meta
          name="description"
          content="Keep your body in tiptop condition with SignatureCare Emergency Center&#039;s emergency room blog and resource center to learn how to avoid common accidents."
        />
        <link rel="canonical" href={mainURL + "/blog"} />
        <link rel="next" href={"/blog/page/" + activePage} />
        <meta property="og:locale" content="en_US" />
        <meta property="og:type" content="article" />
        <meta property="og:title" content="Blog" />
        <meta property="og:url" content={mainURL + "/blog"} />
        <meta
          property="og:site_name"
          content="Emergency Room: SignatureCare Emergency Center 24-Hour ER"
        />
      </Head>

      <div className="bg-dark mb-5 text-white">
        <div className="container text-center">
          <h3 className="py-2">Blog</h3>
        </div>
      </div>

      <section className="container">
        {blogs &&
          blogs.map((blog, index) => (
            <div key={index} className="mb-5">
              <div className="row">
                <div className="col-md-3">
                  {blog.image && (
                    <Image
                      src={laravelURL + "/storage/" + blog.image}
                      alt={blog.title}
                      width={200}
                      height={200}
                      sizes="100vw"
                      className="w-100 object-fit-cover"
                    />
                  )}
                </div>
                <div className="col-md-9">
                  <div className="mt-3">
                    <Link href={"/" + blog.slug}>
                      <h4 className="text-danger mb-3">
                        <b>{blog.title}</b>
                      </h4>
                    </Link>
                    <div
                      dangerouslySetInnerHTML={getDescription(blog.description)}
                    />
                  </div>
                </div>
              </div>
              <div className="py-1 mt-3 border-top border-bottom">
                <small>
                  By <span className="text-danger">SignatureCare ER </span>|{" "}
                  {moment(blog.date).format("MMMM Do, YYYY")}
                  <i> </i>| Categories:{" "}
                  {blog.categories &&
                    blog.categories.map((category, index) => (
                      <span key={index}>
                        {index > 0 ? ", " : ""}
                        <Link
                          href={"/blog/categories/" + category.slug}
                          className="text-danger"
                        >
                          {category.name}
                        </Link>
                      </span>
                    ))}
                </small>
              </div>
            </div>
          ))}

        <div className="d-flex justify-content-center">
          <Pagination
            className="text-danger"
            activePage={activePage}
            itemsCountPerPage={perPage}
            totalItemsCount={total}
            pageRangeDisplayed={5}
            onChange={handlePageChange}
            itemClass="page-item"
            linkClass="page-link"
          />
        </div>
      </section>
    </>
  );
}

export const getServerSideProps = async (context) => {
  const { query } = context;
  const { page } = query;
  const activePage = page ? page : "";
  const res = await http.get(`/blogs/categories?page=${activePage}`);
  const data = await res.data;
  const layoutData = await getLayoutData();

  return {
    props: { blogList: data, layoutData },
  };
};
