import Head from 'next/head';
import { useRouter } from 'next/router';
import parse from 'html-react-parser'
import Image from 'next/image';
import Link from 'next/link';
import { MdPayment } from "react-icons/md";
import { LuCalendarCheck } from "react-icons/lu";
import { FaPhone, FaInfoCircle, FaArrowRight } from "react-icons/fa";
import { BsArrowRightCircleFill } from "react-icons/bs";
import { RiServiceLine } from "react-icons/ri";
import { LiaAwardSolid } from "react-icons/lia";
import http from '../components/utils/http';
import laravelURL from '../components/utils/laravel-url';
import mainURL from '../components/utils/main-url';
import dynamic from 'next/dynamic';
import { getLayoutData } from '../components/utils/getLayoutData'; // ← ADD THIS IMPORT
import { addLazyLoadToImages } from '../components/utils/lazy-images';

const MapHome = dynamic(() => import('../components/templates/map-home'), {
    loading: () => <div style={{ height: '440px' }}>Loading Map...</div>,
});

export default function Home({ page, locations }) {
    const sectionSecond = JSON.parse(page?.second);
    const sectionFive = JSON.parse(page?.fifth);
    const sectionSix = JSON.parse(page?.sixth);
    const router = useRouter();

    if (router.isFallback) {
        return <h1>Loading...</h1>
    }

    return (
        <>
            <Head>
                <title>{page.seo_title}</title>
                <meta name="description" content={page.meta_description} />
                <link rel="canonical" href={mainURL} />

                <meta property="og:title" content={page.seo_title} />
                <meta property="og:description" content={page.meta_description} />
                <meta property="og:url" content={mainURL} />
                <meta property="og:image" content={laravelURL + '/storage/' + page.image} />
 
                {page?.head &&
                    parse(page?.head)
                }
            </Head>

            <div className='position-relative mb-5'>
                {/* Section 1 */}
                <div className="card rounded-0 border-0 section-1">
                    <div className="row g-0">
                        <div className="col-md-6 mb-md-0 mb-1 mobile-height" style={{ minHeight: '600px' }}>
                            <Image
                                src={laravelURL + "/storage/" + page.image}
                                alt='SignatureCare Emergency Center'
                                width={1200}
                                height={600}
                                sizes="(max-width: 768px) 100vw, 50vw"
                                className='w-100 h-100 object-fit-cover'
                                quality={75}
                                priority
                            />
                        </div>
                        <div className="col-md-6 mobile-height" style={{ minHeight: '600px' }}>
                            <MapHome locations={locations} />
                        </div>
                        <div className="fade-overlay d-none d-lg-block"></div>
                    </div>
                </div>
                {/* Section 2 */}
                <div className="card-group shadow container p-0 mx-auto section-2 position-absolute bottom-0 start-50 translate-middle-x">
                    <Link href={sectionSecond[0].link} className="card border-0 border-end bg-white bg-opacity-25 text-white section-2-1">
                        <div className="card-body">
                            <div className='d-flex justify-content-between mb-3'>
                                <h5 className="card-title h4 fw-bold hover-underline">{sectionSecond[0].title}</h5>
                                <div className='text-secondary fs-3'><BsArrowRightCircleFill className='right-circle text-light' /></div>
                            </div>
                            <div className="card-text fs-1"><MdPayment /></div>
                        </div>
                    </Link>
                    <Link href={sectionSecond[1].link} className="card border-0 bg-white bg-opacity-25 text-white section-2-2">
                        <div className="card-body">
                            <div className='d-flex justify-content-between mb-3'>
                                <h5 className="card-title h4 fw-bold hover-underline">{sectionSecond[1].title}</h5>
                                <div className='text-secondary fs-3'><BsArrowRightCircleFill className='right-circle text-light' /></div>
                            </div>
                            <div className="card-text fs-1"><LuCalendarCheck /></div>
                        </div>
                    </Link>
                    <Link href={sectionSecond[2].link} className="card border-0 text-white section-2-3 border-0">
                        <div className="card-body section-2-3-body">
                            <div className='d-flex justify-content-between mb-3'>
                                <h5 className="card-title h4 fw-bold hover-underline">{sectionSecond[2].title}</h5>
                                <div className='text-secondary fs-3'><BsArrowRightCircleFill className='right-circle text-light' /></div>
                            </div>
                            <div className="card-text fs-1"><FaPhone /></div>
                        </div>
                    </Link>
                </div>
            </div >
            {/* Section 2 Small Screen*/}
            <div className="card-group shadow d-lg-none container p-0 mx-auto mb-5" >
                <Link href={sectionSecond[0].link} className="card border-0 border-end text-white section-2-3">
                    <div className="card-body section-2-3-body">
                        <div className='d-flex justify-content-between mb-3'>
                            <h5 className="card-title h4 fw-bold hover-underline">{sectionSecond[0].title}</h5>
                            <div className='text-secondary fs-3'><BsArrowRightCircleFill className='right-circle text-light' /></div>
                        </div>
                        <div className="card-text fs-1"><MdPayment /></div>
                    </div>
                </Link>
                <Link href={sectionSecond[1].link} className="card border-0 text-white section-2-3">
                    <div className="card-body section-2-3-body">
                        <div className='d-flex justify-content-between mb-3'>
                            <h5 className="card-title h4 fw-bold hover-underline">{sectionSecond[1].title}</h5>
                            <div className='text-secondary fs-3'><BsArrowRightCircleFill className='right-circle text-light' /></div>
                        </div>
                        <div className="card-text fs-1"><LuCalendarCheck /></div>
                    </div>
                </Link>
                <Link href={sectionSecond[2].link} className="card border-0 text-white section-2-3 border-0">
                    <div className="card-body section-2-3-body">
                        <div className='d-flex justify-content-between mb-3'>
                            <h5 className="card-title h4 fw-bold hover-underline">{sectionSecond[2].title}</h5>
                            <div className='text-secondary fs-3'><BsArrowRightCircleFill className='right-circle text-light' /></div>
                        </div>
                        <div className="card-text fs-1"><FaPhone /></div>
                    </div>
                </Link>
            </div>
            {/* Section 3 */}
            <div className='container mb-5' >
                <div className='ck-content'>
                    {page?.third && parse(addLazyLoadToImages(page?.third))}
                </div>
            </div>
            {/* Section 4 */}
            <div className='bg-info bg-opacity-10 mb-5' >
                <div className='container text-center'>
                    <button className='btn btn-danger btn-sm rounded-pill mx-2' disabled>
                        <FaInfoCircle /> IMPORTANT INFO </button>
                    <div className='ck-content d-inline-flex mt-3' style={{ margin: '0', padding: '0' }}>
                        {page?.fourth && parse(addLazyLoadToImages(page?.fourth))}
                    </div>
                </div>
            </div>
            {/* Section 5 */}
            <div className='container mb-5 section-5' >
                <div className="row row-cols-1 row-cols-md-3 g-4">
                    {sectionFive?.map((section, index) => (
                        <div className="col" key={index}>
                            <div className="card h-100 p-3">
                                <div className='card-cover-image overflow-hidden'>
                                    <Image
                                        src={laravelURL + "/storage/" + section.image}
                                        alt={section.image.replace('uploads/', '')}
                                        width={400}
                                        height={400}
                                        sizes="(max-width: 768px) 100vw, 33vw"
                                        className="card-img-top w-100 object-fit-cover"
                                        quality={75}
                                        loading="lazy"
                                    />
                                </div>
                                <div className="card-body">
                                    <h3 className="card-title fw-bold">{section.title}</h3>
                                    <p className="card-text">{section.description}</p>
                                </div>
                            </div>
                        </div>
                    ))
                    }
                </div>
            </div>
            <br />
            {/* Section 6 */}
            <div className="card border-0 mb-5">
                <div className="row g-0">
                    <div className="col-md-5">
                        <Image
                            src={laravelURL + "/storage/" + sectionSix.image}
                            alt={sectionSix.heading}
                            width={800}
                            height={600}
                            sizes="(max-width: 768px) 100vw, 50vw"
                            className="img-fluid rounded-end w-100 h-100 object-fit-cover"
                            quality={75}
                            loading="lazy"
                        />
                    </div>
                    <div className="col-md-7 container-sm p-3 p-md-5 bg-info bg-opacity-10">
                        <div className="p-md-5" id="featured-3">
                            <h2 className="pb-2">{sectionSix.heading}</h2>
                            <div>{sectionSix.subheading}</div>
                            <div className="row g-4 py-5 row-cols-1 row-cols-lg-3">
                                {sectionSix?.icons?.map((section, index) => (
                                    <div key={index} className="feature col d-flex align-items-start flex-column bd-highlight">
                                        <div className="feature-icon d-inline-flex align-items-center justify-content-center bg-gradient fs-1 mb-3">
                                            <Image
                                                src={laravelURL + "/storage/" + section.image}
                                                alt={section.image.replace('uploads/', '')}
                                                width={70}
                                                height={70}
                                                sizes="70px"
                                                quality={80}
                                                loading="lazy"
                                            />
                                        </div>
                                        <div className="feature-content">
                                            <h5 className="fw-bold">{section.title}</h5>
                                            <p>{section.description}</p>
                                        </div>
                                        <div className="feature-link mt-auto">
                                            <Link href={section.link} className="icon-link fw-bold d-inline-flex mt-auto">
                                                Read About {section.title} <FaArrowRight className='arrow-right mt-1 ms-1' />
                                            </Link>
                                        </div>
                                    </div>
                                ))}
                            </div>
                            <div className='feature-button fw-bold'>
                                <Link href='/er-services/emergency-care' className='btn btn-outline-info p-3'>
                                    <RiServiceLine /> More Conditions We Treat
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            {/* Section 7 */}
            <div className='container mb-5'>
                <div className='ck-content'>
                    {page?.seventh && parse(addLazyLoadToImages(page?.seventh))}
                </div>
                <div className='feature-button fw-bold'>
                    <Link href='/awards-recognition' className='btn btn-outline-info p-3'>
                        <LiaAwardSolid /> See Our Other Awards
                    </Link>
                </div>
            </div>
            {/* Bottom */}
            {page?.bottom && parse(addLazyLoadToImages(page?.bottom))}
        </>
    )
}

// ← UPDATED getStaticProps - ADD layoutData
export const getStaticProps = async () => {
  try {
    // Fetch all data in parallel with timeout
    const [pageRes, locationsRes, layoutData] = await Promise.allSettled([
      http.get('/admin/home-page'),
      http.get('/admin/locations'),
      getLayoutData(),
    ]);

    // Handle partial failures gracefully
    const page = pageRes.status === 'fulfilled' ? pageRes.value.data : {};
    const locations = locationsRes.status === 'fulfilled' ? locationsRes.value.data : [];
    const layout = layoutData.status === 'fulfilled' ? layoutData.value : {};

    return {
      props: {
        page,
        locations,
        layoutData: layout,
      },
      revalidate: 3600, // 1 hour - reduces backend API calls
    };
  } catch (err) {
    console.error('Error fetching static props:', err.message);
    // Return empty data instead of 404 to prevent blocking
    return {
      props: {
        page: {},
        locations: [],
        layoutData: {},
      },
      revalidate: 60, // Retry after 1 minute if failed
    };
  }
};