import Head from 'next/head';
import Link from 'next/link';
import React, { useRef, useState } from 'react'
import { Form, Row, Col, Button, Spinner } from 'react-bootstrap';
import { ToastContainer, toast } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';
import LocationList from '../components/templates/location-list';
import http from '../components/utils/http';
import mainURL from '../components/utils/main-url';
import ReCAPTCHA from 'react-google-recaptcha';
import { getLayoutData } from "../components/utils/getLayoutData";

export default function InvestorsContact({ locations, locationSidebar }) {
    const [inputs, setInputs] = useState({});
    const [validated, setValidated] = useState(false);
    const [isClicked, setIsClicked] = useState(false);
    const [disabled, setDisabled] = useState(true);
    const recaptchaRef = useRef(null);

    const handleInput = (event) => {
        const name = event.target.name;
        const value = event.target.value;
        setInputs(values => ({ ...values, [name]: value }));
    }

    const recaptHandle = () => {
        setDisabled(false);
    }

    const handleSubmit = (event) => {
        event.preventDefault();
        const form = event.currentTarget;

        if (form.checkValidity() === false) {
            event.stopPropagation();
            setValidated(true);
            return false;
        }

        setIsClicked(true);

        const formData = new FormData();
        formData.append('fname', inputs.fname);
        formData.append('lname', inputs.lname);
        formData.append('phone', inputs.phone);
        formData.append('email', inputs.email);
        formData.append('message', inputs.message);
        formData.append('investor', inputs.investor);

        http.post('/mail/investor-mail', formData).then(res => {
            toast.success(res.data, {
                position: "top-right",
                autoClose: 5000,
                hideProgressBar: false,
                closeOnClick: true,
                pauseOnHover: true,
                draggable: true,
                progress: undefined,
            });
            setInputs({});
            setIsClicked(false);
            setDisabled(true);
            recaptchaRef.current.reset();
            setValidated(false);
        }).catch(error => {
            setIsClicked(false);
        })
    };

    return (
        <>
            <Head>
                <title>Investors | SignatureCare Emergency Centers</title>
                <meta name="description" content='SignatureCare Emergency Center may have investment opportunities for the investor who is seeking a unique growth opportunity.' />
                <link rel="canonical" href={mainURL + "/investors-contact/"} />
                <meta property="og:locale" content="en_US" />
                <meta property="og:title" content='Investors | SignatureCare Emergency Centers' />
                <meta property="og:url" content={mainURL + "/investors-contact/"} />
                <meta property="og:site_name" content="Emergency Room: SignatureCare Emergency Center 24-Hour ER" />
            </Head>

            <ToastContainer />

            <div className='bg-dark text-white'>
                <div className='container text-center'>
                    <h3 className='py-2'>Investors Contact</h3>
                </div>
            </div>
            <br />
            <br />
            <div className='container'>
                <div className='row'>
                    <div className='col-md-9'>
                        <Form noValidate validated={validated} onSubmit={handleSubmit}>
                            <Row className="mb-3">
                                <Form.Group as={Col} md="6" controlId="fname">
                                    <Form.Label>First name</Form.Label>
                                    <Form.Control
                                        name="fname" value={inputs.fname || ''} onChange={handleInput}
                                        required
                                        type="text"
                                        placeholder="First name"
                                    />
                                    <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                                    <Form.Control.Feedback type="invalid">This field is required</Form.Control.Feedback>
                                </Form.Group>
                                <Form.Group as={Col} md="6" controlId="lname">
                                    <Form.Label>Last name</Form.Label>
                                    <Form.Control
                                        name="lname" value={inputs.lname || ''} onChange={handleInput}
                                        required
                                        type="text"
                                        placeholder="Last name"
                                    />
                                    <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                                    <Form.Control.Feedback type="invalid">This field is required</Form.Control.Feedback>
                                </Form.Group>
                            </Row>
                            <Row className="mb-3">
                                <Form.Group as={Col} md="6" controlId="phone">
                                    <Form.Label>Phone Number</Form.Label>
                                    <Form.Control
                                        name="phone" value={inputs.phone || ''} onChange={handleInput}
                                        required
                                        type="text"
                                        placeholder="Phone Number"
                                    />
                                    <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                                    <Form.Control.Feedback type="invalid">This field is required</Form.Control.Feedback>
                                </Form.Group>
                                <Form.Group as={Col} md="6" controlId="email">
                                    <Form.Label>Email</Form.Label>
                                    <Form.Control
                                        name="email" value={inputs.email || ''} onChange={handleInput}
                                        required
                                        type="email"
                                        placeholder="Email"
                                    />
                                    <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                                    <Form.Control.Feedback type="invalid">Invalid email address!</Form.Control.Feedback>
                                </Form.Group>
                            </Row>

                            <Row className="mb-3">
                                <Form.Group as={Col} md="12" controlId="email">
                                    <Form.Label>Are You An Investor Looking For Partnership Opportunity? <span className='text-danger'>*</span></Form.Label>
                                    <Form.Check
                                        label="Yes"
                                        name="investor"
                                        type='radio'
                                        id='reverse-radio-1'
                                        value='Yes' onChange={handleInput} required
                                    />
                                    <Form.Check
                                        label="No"
                                        name="investor"
                                        type='radio'
                                        id='reverse-radio-2'
                                        value='No' onChange={handleInput}
                                    />
                                    <Form.Control.Feedback>Looks good!</Form.Control.Feedback>
                                    <Form.Control.Feedback type="invalid">Invalid email address!</Form.Control.Feedback>
                                </Form.Group>
                            </Row>

                            <Row className="mb-3">
                                <Form.Group as={Col} md="12" className="mb-3" controlId="exampleForm.ControlTextarea1">
                                    <Form.Label>Message <span className='text-danger'>*</span></Form.Label>
                                    <Form.Control name="message" value={inputs.message || ''} onChange={handleInput} as="textarea" rows={5} required placeholder="Type here..." />
                                </Form.Group>
                                <Form.Control.Feedback type="invalid">This field is required</Form.Control.Feedback>
                            </Row>

                            <Form.Group className="mb-3" controlId="formConfirm">
                                <ReCAPTCHA size="normal"
                                    ref={recaptchaRef}
                                    sitekey={process.env.NEXT_PUBLIC_RECAPTCHA_SITE_KEY}
                                    onChange={recaptHandle}
                                />
                            </Form.Group>

                            <div className="d-grid gap-2">
                                {!isClicked &&
                                    <Button disabled={disabled} type="submit" variant="danger" size="lg" className='btn btn-block'>Submit</Button>
                                }

                                {isClicked &&
                                    <Button type="submit" variant="danger" size="lg" className='btn btn-block' disabled>
                                        <Spinner as="span" animation="border" size="sm" role="status" aria-hidden="true" />
                                        <span> Sending...</span>
                                    </Button>
                                }
                            </div>
                        </Form>
                    </div>
                    <div className='col-md-3'>
                        <LocationList locationSidebar={locationSidebar} />
                    </div>
                </div>
                <br />
                <br />
            </div>
        </>
    )
}

export const getStaticProps = async () => {
  try {
    const [locationsRes, sidebarRes, layoutData] = await Promise.all([
      http.get('/admin/locations'),
      http.get('/admin/navigations/location-sidebar'),
      getLayoutData(),
    ]);

    return {
      props: {
        locations: locationsRes.data || [],
        locationSidebar: JSON.parse(sidebarRes.data?.menus || '[]'),
        layoutData,
      },
      revalidate: 10,
    };
  } catch (err) {
    console.error('Error fetching static props:', err.message);
    return { notFound: true };
  }
};
