import React, { useEffect, useRef } from 'react';
import { Button, Card, Form, Alert, FloatingLabel, Col, Spinner } from 'react-bootstrap';
import { useState } from "react";
import http from '../../components/utils/http';
import Head from 'next/head';
import { useRouter } from 'next/router';
import mainURL from '../../components/utils/main-url';
import ReCAPTCHA from 'react-google-recaptcha';
import { getLayoutData } from "../../components/utils/getLayoutData";

export default function ResetPassword() {
    const router = useRouter();
    const [alert, setAlert] = useState(false);
    const [inputs, setInputs] = useState({});
    const [statusMessage, setStatusMessage] = useState('');
    const [errorMessage, setErrorMessage] = useState('');
    const [passCheck, setPassCheck] = useState();
    const [validated, setValidated] = useState(false);
    const [isClicked, setIsClicked] = useState(false);
    const [disabled, setDisabled] = useState(true);
    const recaptchaRef = useRef(null);

    useEffect(() => {
        if (router.isReady) {
            if (!router.query.token) {
                router.push('/membership-login/forgot-password');
            }
            setInputs(values => ({ ...values, email: router.query.email, token: router.query.token }));
        }
    }, [router]);

    const recaptHandle = () => {
        setDisabled(false);
    }

    const handleInput = (event) => {
        const name = event.target.name;
        const value = event.target.value;
        setInputs(values => ({ ...values, [name]: value }));
    }

    const handleSubmit = (event) => {
        const form = event.currentTarget;
        event.preventDefault();
        if (form.checkValidity() === false) {
            event.stopPropagation();
            setValidated(true);
            return false;
        }

        if (inputs.password !== inputs.confirmPassword) {
            setPassCheck(true);
            return;
        }

        setIsClicked(true);
        setErrorMessage('');
        setPassCheck(false);
        setAlert(false);

        http.post("/reset-password", inputs)
            .then((res) => {
                setStatusMessage(res.data.status);
                setAlert(true);
                setIsClicked(false);
                setInputs({});
            }).catch((error) => {
                console.log('error: ' + error)
                setIsClicked(false);
                setAlert(true)
            });
    };

    return (
        <>
            <Head>
                <title>Password Reset for SignatureCare ER Doctors Portal</title>
                <meta name="description" content="Login page for members of SignatureCare Emergency Center Physicians Portal. Please input your username and password to log into the portal." />
                <link rel="canonical" href={mainURL + "/membership-login/"} />
                <meta property="og:locale" content="en_US" />
                <meta property="og:type" content="article" />
                <meta property="og:title" content="Password Reset for SignatureCare ER Doctors Portal" />
                <meta property="og:url" content={mainURL + "/membership-login/"} />
                <meta property="og:site_name" content="Emergency Room: SignatureCare Emergency Center 24-Hour ER" />
                <meta property="article:publisher" content="https://www.facebook.com/SignatureCareER/" />
            </Head>

            <section className='bg-dark text-white'>
                <div className='container text-center'>
                    <h3 className='py-2'>Reset Password</h3>
                </div>
            </section>
            <br />

            <section className='d-flex align-items-center justify-content-center'>
                <Card text='dark' className='w-auto border-0'>
                    <Card.Body>
                        {alert &&
                            <Alert variant="danger" onClick={() => setAlert(false)} dismissible>
                                <b>{statusMessage}</b>
                            </Alert>
                        }

                        <Form noValidate validated={validated} onSubmit={handleSubmit}>
                            <Form.Group as={Col} sm className='mb-3' controlId="Email">
                                <FloatingLabel
                                    controlId="Email"
                                    label="Email..."
                                >
                                    <Form.Control
                                        name="email" value={inputs.email || ''} onChange={handleInput}
                                        type="email"
                                        placeholder="Enter Email"
                                        isInvalid={errorMessage ? true : false}
                                        required
                                        disabled
                                    />
                                </FloatingLabel>
                            </Form.Group>

                            <Form.Group as={Col} sm className='mb-3' controlId="Password">
                                <FloatingLabel
                                    controlId="Password"
                                    label="Password..."
                                >
                                    <Form.Control
                                        name="password" value={inputs.password || ''} onChange={handleInput}
                                        type="password"
                                        isInvalid={passCheck}
                                        required
                                    />
                                </FloatingLabel>
                            </Form.Group>

                            <Form.Group as={Col} sm className='mb-3' controlId="ConfirmPassword">
                                <FloatingLabel
                                    controlId="confirmPassword"
                                    label="Confirm Password..."
                                >
                                    <Form.Control
                                        name="confirmPassword" value={inputs.confirmPassword || ''} onChange={handleInput}
                                        type="password"
                                        isInvalid={passCheck}
                                        required
                                    />
                                </FloatingLabel>
                                <Form.Control.Feedback type="invalid" className={passCheck ? 'd-block' : 'd-none'}>
                                    Password doesnt match.
                                </Form.Control.Feedback>
                            </Form.Group>

                            <Form.Group className="mb-3" controlId="formConfirm">
                                <ReCAPTCHA size="normal"
                                    ref={recaptchaRef}
                                    sitekey={process.env.NEXT_PUBLIC_RECAPTCHA_SITE_KEY}
                                    onChange={recaptHandle}
                                />
                            </Form.Group>

                            {!isClicked &&
                                <Form.Group as={Col}>
                                    <div className='d-grid gap-2'>
                                        <Button disabled={disabled} variant="success" type="submit" className='btn-lg rounded-pill'>
                                            Reset Password
                                        </Button>
                                    </div>
                                </Form.Group>
                            }

                            {isClicked &&
                                <Form.Group as={Col}>
                                    <div className='d-grid gap-2'>
                                        <Button variant="success" size='lg' type="submit" className='rounded-pill' disabled>
                                            <Spinner as="span" animation="border" size="sm" role="status" aria-hidden="true" />
                                            <span> Reset Password...</span>
                                        </Button>
                                    </div>
                                </Form.Group>
                            }
                        </Form>
                    </Card.Body>
                </Card>
            </section>
            <br />
            <br />
            <br />
        </>
    )
}

// ← ADD getServerSideProps (not getStaticProps!)
// Use this for pages with dynamic query parameters or authentication
export async function getServerSideProps(context) {
    try {
        // Fetch layout data
        const layoutData = await getLayoutData();

        return {
            props: {
                layoutData,
            },
        };
    } catch (error) {
        console.error('Error fetching layout data:', error);
        
        return {
            props: {
                layoutData: {},
            },
        };
    }
}