import Head from "next/head";
import Image from "next/image";
import Link from "next/link";
import { useRouter } from "next/router";
import React from "react";
import http from "../components/utils/http";
import laravelURL from "../components/utils/laravel-url";
import mainURL from "../components/utils/main-url";
import SkipTheWait from "../components/utils/skip-the-wait";
import ReactDOMServer from "react-dom/server";
import parse from "html-react-parser";
import { getLayoutData } from "../components/utils/getLayoutData";

export default function ScholarshipPage({ page, scholarships }) {
  const router = useRouter();
  let regex =
    /^.*(?:(?:youtu\.be\/|v\/|vi\/|u\/\w\/|embed\/|shorts\/)|(?:(?:watch)?\?v(?:i)?=|\&v(?:i)?=))([^#\&\?]*).*/;
  const element = <SkipTheWait />;

  if (router.isFallback) {
    return <h1>Loading...</h1>;
  }

  return (
    <>
      <Head>
        <title>{page.seo_title}</title>
        <meta name="description" content={page.meta_description} />
        <link rel="canonical" href={mainURL + "/scholarship"} />

        <meta property="og:title" content={page.seo_title} />
        <meta property="og:description" content={page.meta_description} />
        <meta property="og:url" content={mainURL + "/scholarship"} />
        <meta
          property="og:image"
          content={laravelURL + "/storage/" + page.image}
        />

        {page?.head && parse(page?.head)}
      </Head>

      <div className="bg-dark">
        <div className="container text-center">
          <h3 className="py-2 text-white">{page.title}</h3>
        </div>
      </div>
      <section>
        <div className="progress rounded-0" style={{ height: "10px" }}>
          <div
            className="progress-bar bg-danger"
            role="progressbar"
            style={{ width: "100%" }}
          />
        </div>
        <div className="progress rounded-0" style={{ height: "10px" }}>
          <div
            className="progress-bar bg-dark"
            role="progressbar"
            style={{ width: "100%" }}
          />
        </div>
      </section>
      <br />
      <br />
      <div className="container">
        <div className="row">
          <div className="col-md-9">
            {page?.image && (
              <Image
                src={laravelURL + "/storage/" + page.image}
                alt={page.title}
                width="600"
                height="400"
                sizes="100vw"
                className="w-auto h-auto mb-4"
              />
            )}
            <br />
            <section>
              <div className="ck-content">
                <div
                  dangerouslySetInnerHTML={{
                    __html: page?.description.replace(
                      "skip-the-wait",
                      ReactDOMServer.renderToStaticMarkup(element)
                    ),
                  }}
                />
              </div>
            </section>

            <br />

            <section>
              <h5><b>Some of Our Past Scholarship Winners</b></h5>
              <br />
              {scholarships?.map((scholarship, index) => (
                <div key={index}>
                  {scholarship.year} {scholarship.semester} Winner –{" "}
                  <span>
                    <Link
                      href={scholarship.blog_link || "/scholarship"}
                      target="_blank"
                      className="text-danger"
                    >
                      {scholarship.winner}
                    </Link>
                    (
                    <Link
                      href={scholarship.youtube_link || "/scholarship"}
                      target="_blank"
                      className="text-danger"
                    >
                      See Video
                    </Link>
                    )
                  </span>
                </div>
              ))}
            </section>
          </div>
          <div className="col-md-3">
            <center>
              <b>Some Past Winners</b>
            </center>
            <br />
            <div className="border border-danger">
              {scholarships &&
                scholarships.slice(0, 5).map((scholarship, index) => (
                  <figure key={index}>
                    <Image
                      width="0"
                      height="0"
                      sizes="100vw"
                      style={{ width: "auto", height: "auto" }}
                      src={laravelURL + "/storage/" + scholarship.image}
                      className="w-100 d-block"
                      alt="scholarship-profile"
                    />
                    <figcaption className="text-center">
                      <small>
                        {scholarship.winner} - {scholarship.year}{" "}
                        {scholarship.semester} Semester Winner
                      </small>
                    </figcaption>
                  </figure>
                ))}
            </div>
          </div>
        </div>
        <br />
      </div>
      <section className="bg-dark mb-5">
        <div className="container p-3">
          <center>
            <h3>Recent SignatureCare Emergency Center Scholarship Winners</h3>
          </center>
          <br />
          <div className="row">
            {scholarships?.slice(0, 2).map((scholarship, index) => (
              <div className="col-md-6 mb-3" key={index}>
                {scholarship.youtube_link && (
                  <iframe
                    width="100%"
                    height={315}
                    src={
                      "https://www.youtube.com/embed/" +
                      scholarship.youtube_link.match(regex)[1]
                    }
                  ></iframe>
                )}
              </div>
            ))}
          </div>
        </div>
      </section>

      {page?.bottom && parse(page?.bottom)}
    </>
  );
}

export const getStaticProps = async () => {
  try {
    const [scholarshipsRes, pageRes, layoutData] = await Promise.all([
      http.get("/admin/scholarships"),
      http.get("/admin/pages/scholarship-text"),
      getLayoutData(),
    ]);

    return {
      props: {
        scholarships: scholarshipsRes.data || [],
        page: pageRes.data || {},
        layoutData,
      },
      revalidate: 10,
    };
  } catch (err) {
    console.error("Error fetching scholarships page data:", err.message);
    return { notFound: true };
  }
};
