import React, { useEffect, useState } from 'react'
import moment from 'moment';
import Pagination from 'react-js-pagination';
import { FaSearch } from 'react-icons/fa';
import { Alert } from 'react-bootstrap';
import { useRouter } from 'next/router';
import http from '../components/utils/http';
import Link from 'next/link';
import LocationList from '../components/templates/location-list';
import laravelURL from '../components/utils/laravel-url';
import Image from 'next/image';
import Head from 'next/head';
import mainURL from '../components/utils/main-url';
import { getLayoutData } from "../components/utils/getLayoutData";

export default function Search({ blogList, locationSidebar }) {
    const router = useRouter();
    const { search } = router.query;
    const [blogs, setBlogs] = useState(blogList.data);
    const [activePage, setActivePage] = useState(1);
    const [total, setTotal] = useState(blogList.total);
    const [perPage, setPerPage] = useState(blogList.per_page);
    const [searchInput, setSearchInput] = useState('');
    const [show, setShow] = useState(false);

    const fetchBlogs = async (pageNumber) => {
        const res = await http.get(`/blogs/search?search=${searchInput}&page=${pageNumber}`);
        const data = await res.data;
        setBlogs(data.data);
        setTotal(data.total);
        setPerPage(data.per_page);
        setActivePage(data.current_page);
        router.push(`/search?search=${searchInput}&page=${pageNumber}`, undefined, { shallow: true });
        data.data.length === 0 ? setShow(true) : setShow(false);
    }

    const handleSearch = () => {
        // setActivePage(1);
        // setSearchInput(searchInput);
        fetchBlogs(1);
    }

    const handlePageChange = (pageNumber) => {
        // console.log(`active page is ${pageNumber}`);
        // setActivePage(pageNumber);
        fetchBlogs(pageNumber);
    }

    const getDescription = (description) => {
        return { __html: description.substring(0, 300) };
    }

    return (
        <>
            <Head>
                <title>You searched for {search} - Emergency Room: SignatureCare Emergency Center 24-Hour ER</title>
                <link rel="canonical" href={mainURL + `/search?=` + search} />
                <meta property="og:locale" content="en_US" />
                <meta property="og:type" content="article" />
                <meta property="og:title" content="You searched for signature - Emergency Room: SignatureCare Emergency Center 24-Hour ER" />
                <meta property="og:url" content={mainURL + `/search?=` + search} />
                <meta property="og:site_name" content="Emergency Room: SignatureCare Emergency Center 24-Hour ER" />
            </Head>
            <div className="container">
                <div className='row mt-5'>
                    <div className='col-md-9'>
                        <section>
                            <h3>Need a new search?</h3>
                            <br />
                            <div>If you didn&apos;t find what you were looking for, try a new search!</div>
                            <br />
                            <div className="input-group mb-3">
                                <input value={searchInput} onChange={(e) => setSearchInput(e.target.value)} type="search" className="form-control form-control-lg shadow-none" placeholder="Search..." aria-label="Search..." aria-describedby="Search" autoFocus />
                                <button onClick={handleSearch} className="btn btn-outline-success" type="button" id="Search"><FaSearch /></button>
                            </div>
                            <br />
                            <hr className='curve' />
                            <br />
                        </section>

                        <div className='row'>
                            {blogs &&
                                blogs.map((blog, index) => (
                                    <div className='col-md-4 mb-5 d-flex align-items-stretch' key={index}>
                                        <div className="card">
                                            <Image width="0" height="0" sizes="100vw" style={{ width: 'auto', height: 'auto' }} src={laravelURL + "/storage/" + blog.image} className="card-img-top" alt={blog.title} />
                                            <div className="card-body">
                                                <div className="card-text">
                                                    <Link href={'/blog/' + blog.slug}><h4 className='text-danger mb-3'><b>{blog.title}</b></h4></Link>
                                                    <div className='py-1 mt-3'>
                                                        <small>
                                                            By <span className='text-danger'>SignatureCare ER </span>
                                                            | {moment(blog.date).format("MMMM Do, YYYY")}<i> </i>
                                                            | Categories: {blog.categories &&
                                                                blog.categories.map((category, index) => (
                                                                    <span key={index}>
                                                                        {index > 0 ? ', ' : ''}
                                                                        <Link href={'/blog/categories/' + category.slug} className='text-danger'>{category.name}</Link>
                                                                    </span>
                                                                ))
                                                            }
                                                        </small>
                                                    </div>
                                                    <hr />
                                                    <section>
                                                        {/* <div dangerouslySetInnerHTML={getDescription(blog.description)} /> */}
                                                        <div>{blog.meta_description}</div>
                                                    </section>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                ))
                            }
                        </div>

                        {show &&
                            <Alert variant="danger" onClose={() => setShow(false)} dismissible>
                                <b>Couldn&apos;t find what you&apos;re looking for! Try again</b>
                            </Alert>
                        }

                        <div className='d-flex justify-content-center'>
                            <Pagination className='text-danger'
                                activePage={activePage}
                                itemsCountPerPage={perPage}
                                totalItemsCount={total}
                                pageRangeDisplayed={5}
                                onChange={handlePageChange}
                                itemClass="page-item"
                                linkClass="page-link"
                            />
                        </div>
                    </div>
                    <div className='col-md-3'>
                        <LocationList locationSidebar={locationSidebar} />
                    </div>
                </div>
            </div>
            <br />
            <br />
            <br />
        </>
    )
}

export const getServerSideProps = async (context) => {
    const { query } = context;
    const { search } = query
    const { page } = query
    const queryString = search ? search : '';
    const activePage = page ? page : '';
    const url = `/blogs/search?search=${queryString}&page=${activePage}`;
    const res = await http.get(url);
    const data = await res.data;
    // const location = await http.get('/admin/locationSidebar');
    // const locations = await location.data;
    const locationList = await http.get('/admin/navigations/location-sidebar');
    const locationSidebar = await locationList.data;
    const layoutData = await getLayoutData();

    return {
        props: {
            blogList: data,
            // locations: locations,
            locationSidebar: JSON.parse(locationSidebar.menus),
            layoutData,
        },
    }
}
